<?php

namespace App\Http\Controllers;

use App\Bank;
use App\CustomerCase;
use App\Image;
use App\Traits\ApiResponse;
use App\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use function GuzzleHttp\Promise\all;

class CustomerController extends Controller
{
    use ApiResponse;

    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'mobile' => 'required|unique:customers,mobile',
            'cnic' => 'required',
            'email' => 'required|unique:customers,email',
            'shipping_address' => 'required',
            'user_type' => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $customer = Customer::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => bcrypt($request->password),
                'user_type' => $request->user_type,
                'cnic' => $request->cnic,
                'mobile' => $request->mobile,
                'shipping_address' => $request->shipping_address,
                'office_address' => $request->office_address,
                'account_no' => $request->account_no,
                'courts' => $request->courts,
                'account_title' => $request->account_title,
                'bank_account_title' => $request->bank_account_title,
                'bank_name' => $request->bank_name,
                'bank_account_no' => $request->bank_account_no,
                'status' => 1,
            ]);
            if($request->user_type === 'attorney'){
                if($request->has('id_card_front')){
                    $imageUrl = $this->addUserImageData($request->id_card_front);
                    Image::create([
                        'imageable_type' => get_class(new Customer()),
                        'imageable_id' => $customer->id,
                        'url' => $imageUrl,
                        'type' => 'id card front',
                    ]);
                }
                if($request->has('id_card_back')){
                    $imageUrl = $this->addUserImageData($request->id_card_back);
                    Image::create([
                        'imageable_type' => get_class(new Customer()),
                        'imageable_id' => $customer->id,
                        'url' => $imageUrl,
                        'type' => 'id card back',
                    ]);
                }
                if($request->has('clerk_card')){
                    $imageUrl = $this->addUserImageData($request->clerk_card);
                    Image::create([
                        'imageable_type' => get_class(new Customer()),
                        'imageable_id' => $customer->id,
                        'url' => $imageUrl,
                        'type' => 'clerk card',
                    ]);
                }
            }
        }

        $customer['token'] = $this->createUserToken($customer);

        $message  = 'You have successfully provided required details for Registration as Attorney and your registration is under review, you will receive confirmation email on your given registered email address within 48 Hours. After that you can login to Attorney Official and work as an attorney.';

        return $this->success(200, $request->user_type === 'customer'? $customer : null, $request->user_type === 'customer'? 'User registered successfully' : $message);
    }

    public function login(Request $request)
    {
        $credentials = $request->only('mobile', 'password');

        //valid credential
        $validator = Validator::make($credentials, [
            'mobile' => 'required',
            'password' => 'required|string|min:6|max:50'
        ]);

        //Send failed response if request is not valid
        if ($validator->fails()) {
            return response()->json(['error' => $validator->messages()], 200);
        }

        $customer = Customer::where('mobile', $request->mobile)->first();
        if ($customer) {
            if($customer->user_type === 'attorney'){
                if($customer->status === 2){
                    if (Hash::check($request->password, $customer->password)) {
                        $customer['token'] = $this->createUserToken($customer);
                        return $this->success(200, $customer, 'user login successfully');
                    }
                    else{
                        return $this->error(400, null, 'Incorrect mobile/password');
                    }
                }
                else{
                    return $this->success(200, $customer, 'You are not verified attorney');
                }
            }
            if($customer->user_type === 'customer'){
                if (Hash::check($request->password, $customer->password)) {
                    $paymentMethodCount = $this->getPaymentMethodCount();
                    $customer['pendingCaseCounts'] =  $this->getCustomerCasesCount($customer->id, 1);
                    $customer['inProgressCaseCounts'] =  $this->getCustomerCasesCount($customer->id, 2);
                    $customer['completedCaseCounts'] =  $this->getCustomerCasesCount($customer->id, 3);
                    $customer['paymentMethodCount'] = $paymentMethodCount;
                    $customer['token'] = $this->createUserToken($customer);
                    $customer['address'] = Config::get('ADMIN_OFFICE_ADDRESS');
                    $customer['whatsapp'] = Config::get('ADMIN_WHATSAPP');
                    return $this->success(200, $customer, 'user login successfully');
                }
                else{
                    return $this->error(400, null, 'Incorrect mobile/password');
                }
            }
        }
        else{
            return $this->error(400, null, 'Incorrect mobile/password');
        }
    }

    public function logout(Request $request)
    {
        //valid credential
        $validator = Validator::make($request->only('token'), [
            'token' => 'required'
        ]);

        //Send failed response if request is not valid
        if ($validator->fails()) {
            return response()->json(['error' => $validator->messages()], 200);
        }

        //Request is validated, do logout
        try {
            JWTAuth::invalidate($request->token);

            return response()->json([
                'success' => true,
                'message' => 'User has been logged out'
            ]);
        } catch (JWTException $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Sorry, user cannot be logged out'
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    public function createUserToken($user)
    {
        return 'Bearer ' . $user->createToken('attorney')->accessToken;
    }

    public function update(Request $request){
        $customer = Customer::findOrFail(Auth::id());
        if($customer)
        {
            $data = $request->password ? $request->all() : $request->except('password');
            $customer->account_title = null;
            $customer->account_no = null;
            $customer->bank_account_title = null;
            $customer->bank_account_no = null;
            $customer->bank_name = null;
            $customer->update($data);
            return $this->success(200, $customer, 'user updated successfully');
        }
        else{
            return $this->error(400, '', 'user not found');
        }
    }

    public function addUserImageData($requestFile){
            $image_uploaded_path = $requestFile->store('images', 'public');
            $imageUrl = Storage::disk('public')->url($image_uploaded_path);
            return $imageUrl;
    }

    public function getCustomerCasesCount($customerId, $type){
        return CustomerCase::where('customer_id', $customerId)
            ->groupBy('status')->where('status', $type)
                ->count();
    }

    public function getPaymentMethodCount(){
        return Bank::where('status', 1)->get()->count();
    }

    public function updatePassword(Request $request){
        $customer = Customer::where('mobile', $request->mobile)->first();
        if($customer){
            Customer::where('mobile', $request->mobile)->update([
                'password' => bcrypt($request->password),
            ]);
            return $this->success(200, null, 'Your password has been updated successfully');
        }
        else{
            return $this->error(400, null, 'Incorrect mobile number');
        }

    }

    public function updateFcmToken(Request $request){
        $customer = Customer::findOrFail(Auth::id());
        if($customer)
        {
            Customer::where('id', Auth::id())->update([
                'fcm_token' => $request->fcm_token,
            ]);
        }
    }

    public function verifyUser(Request $request){
        $customer = Customer::where('mobile', $request->mobile)->first();
        if($customer){
            Customer::where('mobile', $request->mobile)->update([
                'status' => 1,
            ]);
            return $this->success(200, ['user' => $customer->user_type], 'User verified successfully');
        }
        else{
            return $this->error(400, null, 'User not verified');
        }
    }

    public function getUpdatedCounts(){
        $customerId = Auth::id();
        $paymentMethodCount = $this->getPaymentMethodCount();
        $customer['pendingCaseCounts'] =  $this->getCustomerCasesCount($customerId, 1);
        $customer['inProgressCaseCounts'] =  $this->getCustomerCasesCount($customerId, 2);
        $customer['completedCaseCounts'] =  $this->getCustomerCasesCount($customerId, 3);
        $customer['paymentMethodCount'] = $paymentMethodCount;
        $customer['address'] = Config::get('ADMIN_OFFICE_ADDRESS');
        $customer['whatsapp'] = Config::get('ADMIN_WHATSAPP');
        return $this->success(200, $customer, 'user login successfully');
    }

}
