<?php

namespace App\Http\Controllers;

use App\CaseCost;
use App\CaseDetail;
use App\CasePayment;
use App\Customer;
use App\CustomerCase;
use App\District;
use App\Image;
use App\Traits\ApiResponse;
use App\Traits\PushNotification;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CustomerCaseController extends Controller
{

    use ApiResponse;

    public function generateCaseRequest(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_type' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::create([
                'customer_id' => Auth::id(),
                'attorney_id' => $request->attorney_id,
                'case_type' => $request->case_type,
                'case_status' => $request->case_status,
                'court_name' => $request->court_name,
                'news_paper_name' => $request->news_paper_name,
                'city' => $request->city,
                'title' => $request->title,
                'judge_name' => $request->judge_name,
                'case_no' => $request->case_no,
                'year' => $request->year,
                'decision_date' => $request->decision_date,
                'last_date_hearing' => $request->last_date_hearing,
                'next_hearing_date' => $request->next_hearing_date,
                'required_document' => $request->required_document,
                'from_which_side_we_are' => $request->from_which_side_we_are,
                'set_of_copies_required' => $request->set_of_copies_required,
                'kuliya_no' => $request->kuliya_no,
                'parties_name' => $request->parties_name,
                'comment_for_attorney' => $request->comment_for_attorney,
                'no_of_defendant' => $request->no_of_defendant,
                'serial_no_of_defendant' => $request->serial_no_of_defendant,
            ]);
            $castCostDetails = CaseCost::where('case_type', $request->case_type)->first();


            if ($request->case_type == 9) {
                $amount = $request->no_of_defendant * $castCostDetails->per_defendant_cost;
//                $message = 'Kindly transfer Rs. ' . $amount . '/-  in official bank accounts of Attorney Official as provided in the Payment Method options in main screen of the app. Per Person Summon/Notice Charges is Rs. 300/- and according to your number of persons total charges are shown to you.';
                $message = 'Per person summon/notice charges are Rs. ' . $castCostDetails->per_defendant_cost . '/- and according to your mentioned number of persons your total summon/notices charges are Rs. ' . $amount . '/-. Kindly Pay Rs. ' . $amount . '/- in the Easypaisa account of the Attorney Official. ';
            } else if ($request->case_type == 10) {
                $message = 'Kindly wait your order of Newspaper Advertisement will be marked shortly to the Attorney, who will update you regarding charges of your Newspaper Advertisement.';
                $amount = 0;
            } else {
                if ($request->case_status == 2) {
                    $message = $castCostDetails->initial_decided_message;
                    $amount = $castCostDetails->initial_decided_cost;
                } else {

                    $message = $castCostDetails->initial_pending_message;
                    $amount = $castCostDetails->initial_pending_cost;

                }
            }

            $caseDetails = CaseDetail::create([
                'case_id' => $case->id,
                'user_id' => Auth::id(),
                'case_status' => 'case request generated',
                'type' => 'request',
                'amount' => $amount,
                'message' => $message,
                'user_type' => 'attorney',
            ]);

            if ($case->case_type === '9' || $case->case_type === '10') {
                if ($request->has('case_images')) {
                    foreach ($request->case_images as $image) {
                        $imageUrl = $this->addCaseImageData($image);
                        Image::create([
                            'imageable_type' => get_class(new CaseDetail()),
                            'imageable_id' => $caseDetails->id,
                            'url' => $imageUrl,
                            'type' => 'summon_news_paper_case_images',
                        ]);
                    }
                }
            }

            $case['caseDetails'] = $caseDetails;

            return $this->success(200, $case, $message);
        }
    }

    public function addCaseImageData($requestFile)
    {
        $image_uploaded_path = $requestFile->store('images', 'public');
        $imageUrl = Storage::url($image_uploaded_path);
        $imageUrl = env('APP_URL') . '/public' . $imageUrl;
        return $imageUrl;
    }

    public function updateCaseDetails($caseId, $user, $userType, $caseStatus, $type, $message = null, $paymentId = null, $amount = null)
    {
        $caseDetails = CaseDetail::create([
            'case_id' => $caseId,
            'user_id' => $user,
            'user_type' => $userType,
            'case_status' => $caseStatus,
            'type' => $type,
            'message' => $message,
            'payment_id' => $paymentId,
            'amount' => $amount,
        ]);
        return $caseDetails;
    }

    public function updateCaseStatus($caseId, $status)
    {
        CustomerCase::where('id', $caseId)->update(['status' => $status]);
    }

    public function getAllCases($type)
    {
        $customerCases = CustomerCase::with('customerCaseDetails', 'caseImages', 'attorneyDetails', 'customerDetails')
            ->where('customer_id', Auth::id())
            ->where('status', $type)
            ->orderBy('id', 'desc')
            ->get();

        if ($customerCases && count($customerCases) > 0) {
            foreach ($customerCases as $case) {
                $case->case_type_name = Config::get('constants.cases_type.' . $case->case_type);
            }
            return $this->success(200, $customerCases, '');
        } else {
            return $this->success(200, null, 'No Case found');
        }
    }

    public function getCaseById(Request $request)
    {
        $customerCases = CustomerCase::with('customerCaseDetails', 'customerCaseDetails.caseDetailImage', 'customerCaseDetails.caseDetailMultipleImages', 'caseImages', 'attorneyDetails', 'customerDetails')
            ->where('customer_id', Auth::id())
            ->where('id', $request->case_id)
            ->first();
        if ($customerCases && $customerCases->case_type == 10 && $this->checkCasePaymentDetailsNewspaper($customerCases)){
            $customerCases = CustomerCase::with('customerCaseDetails', 'customerCaseDetails.caseDetailImage', 'customerCaseDetails.caseDetailMultipleImages', 'caseImages', 'attorneyDetails', 'customerDetails')
            ->where('customer_id', Auth::id())
                ->where('id', $request->case_id)
                ->first();
        }
        else if ($this->checkCasePaymentDetails($customerCases)){
            $customerCases = CustomerCase::with('customerCaseDetails', 'customerCaseDetails.caseDetailImage', 'customerCaseDetails.caseDetailMultipleImages', 'caseImages', 'attorneyDetails', 'customerDetails')
            ->where('customer_id', Auth::id())
                ->where('id', $request->case_id)
                ->first();
        }


        if ($customerCases) {
            $customerCases->case_type_name = Config::get('constants.cases_type.' . $customerCases->case_type);
            return $this->success(200, $customerCases, '');
        } else {
            return $this->success(200, null, 'No Case found');
        }
    }

    public function checkCasePaymentDetails($customerCases)
    {
        if ($customerCases && $customerCases['customerCaseDetails']) {
            if (count($customerCases['customerCaseDetails']) > 0) {
                $caseDetail = $customerCases['customerCaseDetails'][count($customerCases['customerCaseDetails']) - 1];
                if ($caseDetail->type == 'request' || $caseDetail->type == 'attorney_case_pages_update' || $caseDetail->type == 'attorney_case_update'
                    && (int)$caseDetail->amount > 0) {
                    $casePayment = CasePayment::where(['case_id' => $caseDetail->id, 'response_code' => '0000', 'amount' => $caseDetail->amount])->orderBy('id', 'desc')->first();
                    if ($casePayment) {
                        $this->casePaymentUpdate($customerCases->id, $caseDetail->type == 'request' ? 'initial' : 'case_payment', $casePayment->id);
                        return true;
                    }
                    else {
                        $payload = [
                            "orderId" => $caseDetail->id,
                            "storeId" => "417746",
                            "accountNum" => '142143489'
                        ];
                        $response = Http::withHeaders(['Credentials' => env('EASE_PAISA_CREDENTIALS_URL')])
                            ->post(env('EASE_PAISA_INQUIRE_LIVE_URL'), $payload);

                        if ($response->status() == 200) {
                            $responseBody = $response->body();
                            $data = \GuzzleHttp\json_decode($responseBody);
                            $data2 = \GuzzleHttp\json_decode($responseBody, true);

                            if ($data->responseCode === '0000'){
                                if(array_key_exists('errorCode', $data2) && array_key_exists('transactionStatus', $data2) && $data2['transactionStatus'] === "FAILED" ){
                                    return false;
                                }
                                if ($data->transactionAmount == $caseDetail->amount){
                                    if ($data->responseCode == '0000'){
                                        $casePayment = new CasePayment();
                                        $casePayment->case_id = $data->orderId;
                                        $casePayment->transaction_id = $data->transactionId ?? '0';
                                        $casePayment->transaction_date_time = $data->transactionDateTime;
                                        $casePayment->amount = $data->transactionAmount;
                                        $casePayment->response_code = $data->responseCode;
                                        $casePayment->response_desc = $data->responseDesc;
                                        $casePayment->save();

                                        $this->casePaymentUpdate($customerCases->id, $caseDetail->type == 'request' ? 'initial' : 'case_payment', $casePayment->id);
                                        return true;
                                    }
                                }
                            }
                        }
                    }
                    return false;
                }
                return false;
            }
            return false;
        }
    }
    public function checkCasePaymentDetailsNewspaper($customerCases)
    {
        if ($customerCases && $customerCases['customerCaseDetails']) {
            if (count($customerCases['customerCaseDetails']) > 0) {
                $caseDetail = $customerCases['customerCaseDetails'][count($customerCases['customerCaseDetails']) - 1];

                if ($caseDetail->type == 'attorney_case_news_paper_charges_update' && (int)$caseDetail->amount > 0) {
                    $casePayment = CasePayment::where(['case_id' => $caseDetail->id, 'response_code' => '0000', 'amount' => $caseDetail->amount])->orderBy('id', 'desc')->first();

                    if ($casePayment) {
                        $this->casePaymentUpdate($customerCases->id, 'submit_ad_in_news_agency', $casePayment->id);
                        return true;
                    }
                    else {
                        $payload = [
                            "orderId" => $caseDetail->id,
                            "storeId" => "417746",
                            "accountNum" => '142143489'
                        ];
                        $response = Http::withHeaders(['Credentials' => env('EASE_PAISA_CREDENTIALS_URL')])
                            ->post(env('EASE_PAISA_INQUIRE_LIVE_URL'), $payload);

                        if ($response->status() == 200) {
                            $responseBody = $response->body();
                            $data = \GuzzleHttp\json_decode($responseBody);
                            $data2 = \GuzzleHttp\json_decode($responseBody, true);

                            if ($data->responseCode === '0000'){
                                if(array_key_exists('errorCode', $data2) && array_key_exists('transactionStatus', $data2) && $data2['transactionStatus'] === "FAILED" ){
                                    return false;
                                }
                                if ($data->transactionAmount == $caseDetail->amount){
                                    if ($data->responseCode == '0000'){
                                        $casePayment = new CasePayment();
                                        $casePayment->case_id = $data->orderId;
                                        $casePayment->transaction_id = $data->transactionId ?? '0';
                                        $casePayment->transaction_date_time = $data->transactionDateTime;
                                        $casePayment->amount = $data->transactionAmount;
                                        $casePayment->response_code = $data->responseCode;
                                        $casePayment->response_desc = $data->responseDesc;
                                        $casePayment->save();

                                        $this->casePaymentUpdate($customerCases->id, $caseDetail->type == 'request' ? 'initial' : 'case_payment', $casePayment->id);
                                        return true;
                                    }
                                }
                            }
                        }
                    }
                    return false;
                }
                return false;
            }
            return false;
        }
    }

    public function casePaymentUpdate($caseId, $paymentType, $paymentId)
    {
        $case = CustomerCase::where('id', $caseId)->first();


        if ($case->case_type == 9) {
            $imageType = 'case_payment';
            $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will serve summon/notice as per procedure of the court.';
        } else if ($case->case_type == 10) {
            $imageType = 'charges_payment';
            $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and complete your order of Newspaper Advertisement before next date of hearing of your case as per court order. You can see your Newspaper Advertisement on next date of hearing of your case and Newspaper will also be submitted before the court in your case file.';
        } else {
            if ($paymentType === 'initial' && $case->case_type == 7) {
                $imageType = 'initial_payment';
//                $message = 'Your order of ' . Config::get('constants.cases_type.' . $case->case_type) . ' will be marked shortly to the attorney and in Rs. ' . Config::get('constants.cases_type_amount.' . $case->case_type) . '/- the attorney would check your file and update you soon';
                $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, whose office address will be shared shortly for sending your case file through the courier service. Meanwhile, kindly complete your case file by fulfilling all the requirements of filing your case';

            }
            else if ($paymentType === 'initial') {

                $imageType = 'initial_payment';
//                $message = 'Your order of ' . Config::get('constants.cases_type.' . $case->case_type) . ' will be marked shortly to the attorney and in Rs. ' . Config::get('constants.cases_type_amount.' . $case->case_type) . '/- the attorney would check your file and update you soon';
                if($case->case_status == 'Pending' && $case->case_type != 4){
                    $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will update you soon regarding the cost of ' . Config::get('constants.cases_type.' . $case->case_type) .' and service charges';
                }
                elseif($case->case_status == 'Decided'){
                    $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will update you soon regarding the cost of ' . Config::get('constants.cases_type.' . $case->case_type) .' and service charges';
                }
                else{
                    $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will update you soon.';
                }

            } else if ($paymentType === 'case_payment') {
                $imageType = 'case_payment';
                $message = 'You have paid the amount Successfully.';
            }
        }

            $this->updateCaseStatus($caseId, 2);
            $caseDetails = $this->updateCaseDetails($caseId, Auth::id(),
                'customer', 'payment successfully', $paymentType, $message, $paymentId);

            PushNotification::sendPushNotification(Config::get('constants.push_notification_customer_update_title'), $message, $case->attorney_id);
            return $this->success(200, null, 'Payment has been successful');
    }

    public function getCaseAssignedToAttorneyById(Request $request)
    {
        $customerCases = CustomerCase::with('customerCaseDetails', 'customerCaseDetails.caseDetailImage',
            'customerCaseDetails.caseDetailMultipleImages', 'caseImages', 'attorneyDetails', 'customerDetails', 'getPaymentDetails')
            ->where('attorney_id', Auth::id())
            ->where('id', $request->case_id)
            ->first();
        if ($customerCases) {
            $customerCases->case_type_name = Config::get('constants.cases_type.' . $customerCases->case_type);

            if (count($customerCases['getPaymentDetails']) > 0) {
                $count = 0;
                foreach ($customerCases['getPaymentDetails'] as $paymentDetail) {
                    if ($paymentDetail->transaction_id != 0) {
                        $count += (int)$paymentDetail->amount;
                    }
                }
                $customerCases->amountReceived = $count;

                if ($customerCases->case_type == 4) {
                    $amount = (((double)Config::get('constants.case_charges.high_court_case_services_charges') +
                                ($customerCases->case_status == 'Pending' ? (double)Config::get('constants.case_charges.initial_high_court_charges') : 1000)
                            ) *
                            (double)Config::get('constants.AMOUNT_PERCENTAGE')) + ($count -
                            ((double)Config::get('constants.case_charges.high_court_case_services_charges') +
                                ($customerCases->case_status == 'Pending' ? (double)Config::get('constants.case_charges.initial_high_court_charges') : 1000)));
                    $customerCases->attorneyAmountReceived = $amount > 0 ? floor($amount) : 0;
                } else if ($customerCases->case_type == 1 || $customerCases->case_type == 2 || $customerCases->case_type == 3) {
                    $amount =
                        ((($customerCases->case_status == 'Pending' ? (double)Config::get('constants.case_charges.case_services_charges') : 1399) +
                                ($customerCases->case_status == 'Pending' ? (double)Config::get('constants.case_charges.initial_charges') : 1000)
                            ) *
                            (double)Config::get('constants.AMOUNT_PERCENTAGE')) +
                        ($count - (($customerCases->case_status == 'Pending' ? (double)Config::get('constants.case_charges.case_services_charges') : 1399) +
                                ($customerCases->case_status == 'Pending' ? (double)Config::get('constants.case_charges.initial_charges') : 1000)));

                    $customerCases->attorneyAmountReceived = $amount > 0 ? floor($amount) : 0;

                } else {
                    $amount = $count * (double)Config::get('constants.AMOUNT_PERCENTAGE');
                    $customerCases->attorneyAmountReceived = $amount > 0 ? floor($amount) : 0;
                }
            }

            return $this->success(200, $customerCases, '');
        } else {
            return $this->success(200, null, 'No Case found');
        }
    }

    public function getAllCasesByAttorney($type)
    {
        $customerCases = CustomerCase::with('customerCaseDetails', 'caseImages', 'attorneyDetails', 'customerDetails')
            ->where('status', $type)
            ->where('attorney_id', Auth::id())->orderBy('id', 'desc')->get();

        if ($customerCases && count($customerCases) > 0) {
            foreach ($customerCases as $case) {
                $case->case_type_name = Config::get('constants.cases_type.' . $case->case_type);
            }
            return $this->success(200, $customerCases, '');
        } else {
            return $this->success(200, null, 'No Case found');
        }
    }

    public function casePayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
            'payment_type' => 'required',
        ]);
        $case = CustomerCase::where('id', $request->case_id)->first();

        if ($case->case_type == 9) {
            $imageType = 'case_payment';
            $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will serve summon/notice as per procedure of the court.';
        } else if ($case->case_type == 10) {
            $imageType = 'charges_payment';
            $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and complete your order of Newspaper Advertisement before next date of hearing of your case as per court order. You can see your Newspaper Advertisement on next date of hearing of your case and Newspaper will also be submitted before the court in your case file.';
        } else {
            if ($request->payment_type === 'initial' && $case->case_type == 7) {
                $imageType = 'initial_payment';
//                $message = 'Your order of ' . Config::get('constants.cases_type.' . $case->case_type) . ' will be marked shortly to the attorney and in Rs. ' . Config::get('constants.cases_type_amount.' . $case->case_type) . '/- the attorney would check your file and update you soon';
                $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, whose office address will be shared shortly for sending your case file through the courier service. Meanwhile, kindly complete your case file by fulfilling all the requirements of filing your case';

            }
            else if ($request->payment_type === 'initial') {
                $imageType = 'initial_payment';
//                $message = 'Your order of ' . Config::get('constants.cases_type.' . $case->case_type) . ' will be marked shortly to the attorney and in Rs. ' . Config::get('constants.cases_type_amount.' . $case->case_type) . '/- the attorney would check your file and update you soon';
                if($case->case_status == 'Pending' && $case->case_type != 4){
                    $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will update you soon regarding the cost of ' . Config::get('constants.cases_type.' . $case->case_type) .' and service charges';
                }
                elseif($case->case_status == 'Decided'){
                    $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will update you soon regarding the cost of ' . Config::get('constants.cases_type.' . $case->case_type) .' and service charges';
                }
                else{
                    $message = 'You have paid the amount Successfully. Attorney Official will verify your payment and mark your case shortly to the Attorney, who will update you soon.';
                }

            } else if ($request->payment_type === 'case_payment') {
                $imageType = 'case_payment';
                $message = 'You have paid the amount Successfully.';
            }
        }

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $this->updateCaseStatus($request->case_id, 2);
            $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(),
                'customer', 'payment successfully', $request->payment_type, $message, $request->payment_id);
            if ($request->has('payment_image')) {
                $imageUrl = $this->addCaseImageData($request->payment_image);
                Image::create([
                    'imageable_type' => get_class(new CaseDetail()),
                    'imageable_id' => $caseDetails->id,
                    'url' => $imageUrl,
                    'type' => $imageType,
                ]);
            }
            PushNotification::sendPushNotification(Config::get('constants.push_notification_customer_update_title'), $message, $case->attorney_id);
            return $this->success(200, null, 'Payment has been successful');
        }
    }

    public function markCaseCompletedByAttorney(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();
            if ($case->case_type == 10) {
                $this->updateCaseStatus($request->case_id, 3);
                $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(),
                    'attorney', 'completed', 'case_processed', 'Case processed successfully.');
                $message = 'Attorney has completed your newspaper advertisement service and you can see your newspaper advertisement in your case file on next date of hearing.';
                PushNotification::sendPushNotification('News paper Ad Update', $message, $case->customer_id);
                return $this->success(200, null, $message);
            } else {
                $this->updateCaseStatus($request->case_id, 3);
                $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(),
                    'attorney', 'completed', 'case_processed', 'Case processed successfully.');
                $message = 'Case processed successfully.';
                PushNotification::sendPushNotification(Config::get('constants.push_notification_customer_completed_title'), $message, $case->customer_id);
                return $this->success(200, null, $message);
            }
        }
    }

    public function attorneyReceivedFileFromCustomer(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();
            $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(),
                'attorney', 'received file from customer', 'filing_any_application_received_file_from_customer',
                'Attorney has received your application and will submit your application before the court where your case is pending.');
            $message = 'Attorney has received your application and will submit your application before the court where your case is pending.';
            PushNotification::sendPushNotification(Config::get('constants.push_notification_customer_update_title'), $message, $case->customer_id);
            return $this->success(200, null, 'Case processed successfully.');
        }
    }

    public function attorneySubmitNewsPaperAd(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();
            $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(),
                'attorney', 'submit ad in news agency', 'submit_ad_in_news_agency',
                'Attorney has submitted your Ad in News Paper');
            $message = 'Attorney has submitted your Ad in News Paper';
            PushNotification::sendPushNotification(Config::get('constants.push_notification_customer_completed_title'), $message, $case->customer_id);
            return $this->success(200, null, 'Notify customer successfully');
        }
    }

    public function notifyAttorneyForFileCase(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();
            if ($case->case_type == 7) {
                $this->updateCaseDetails($request->case_id, Auth::id(),
                    'customer', 'file sent to attorney', 'file_sent_to_attorney', 'File sent to attorney through Courier Service Successfully.');
            } else if ($case->case_type == 8) {
                $this->updateCaseDetails($request->case_id, Auth::id(),
                    'customer', 'file sent to attorney', 'file_any_application_sent_to_attorney', 'File sent to attorney through Courier Service Successfully.');
            }
            $message = 'File sent to attorney through Courier Service Successfully.';
            PushNotification::sendPushNotification(Config::get('constants.push_notification_customer_update_title'), $message, $case->attorney_id);
            return $this->success(200, null, ' Attorney notified successfully');
        }
    }

    public function attorneyUpdateCaseDetails(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $message = '';
            $caseCustomer = CustomerCase::where('id', $request->case_id)->first();
            if ($request->has('hearing_date')) {
                $message = 'Next date of hearing of your case is ' . $request->hearing_date;
                $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about case hearing date',
                    'attorney_case_update_of_hearing', $message);
                $this->updateCaseStatus($request->case_id, 2);
            } else if ($request->has('interim_order_image')) {
                $message = 'Please check your interim order image';
                $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about interim order',
                    'attorney_case_update_interim_order', $message);
                $this->updateCaseStatus($request->case_id, 2);
                if ($request->has('interim_order_image')) {
                    foreach ($request->interim_order_image as $image) {
                        $imageUrl = $this->addCaseImageData($image);
                        Image::create([
                            'imageable_type' => get_class(new CaseDetail()),
                            'imageable_id' => $caseDetails->id,
                            'url' => $imageUrl,
                            'type' => 'interim_order',
                        ]);
                    }
                }
            } else if ($request->has('filed_case_image')) {
                $message = 'Your Case has been filed by the attorney successfully. Your case is entrusted to ' . $request->judge_name . ' and your next date of hearing of your filed case is ' . $request->next_hearing_date;
                $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about filed case',
                    'attorney_case_update_filed', $message);
                CustomerCase::where('id', $request->case_id)->update(['judge_name' => $request->judge_name]);
            } else if ($request->has('receipt_case_image')) {
                $message = 'Attorney has served the summon/notice as per order of the court and will also submit the original receipts of the courier service on the date of hearing of your case before the court.';
                $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about summon notice',
                    'attorney_case_update_summon', $message);
            } else {
                $case = CustomerCase::where('id', $request->case_id)->first();
                if ($case->case_type == 1 || $case->case_type == 2 || $case->case_type == 3 || $case->case_type == 4) {
                    $castCostDetails = CaseCost::where('case_type', $case->case_type)->first();
                    if ($case->case_type == 4) {
                        if ($case->case_status == 'Pending') {
                            $serviceCharges = $castCostDetails->pending_service_charges;
                        } else {
                            $serviceCharges = $castCostDetails->decided_service_charges;
                        }
                    } else {
                        if ($case->case_status == 'Pending') {
                            $serviceCharges = $castCostDetails->pending_service_charges;
                        } else {
                            $serviceCharges = $castCostDetails->decided_service_charges;
                        }
                    }
                    $perPageCharges = $castCostDetails->per_page_cost;
                    $pagesCharges = $perPageCharges * $request->pages;
//                    dd($perPageCharges, $pagesCharges);

//                    $message = 'Your per page cost of ' . Config::get('constants.cases_type.' . $case->case_type) . ' including Indirect Charges is Rs. ' . $perPageCharges . '/- Total Number of your Attested Copy are ' . $request->pages . ' pages Total Cost of Copies is Rs. ' . $pagesCharges . '/- Service Charges Rs. ' . $serviceCharges . '/- Total Charges are ' . $serviceCharges . ' + ' . $pagesCharges . ' = ' . $chargesWithOutDelivery . '/-';
                    $message = 'Total number of pages of your attested copy are ' . $request->pages . ' Pages. Attested Copy Per Page Charges is Rs. ' . $perPageCharges . '/- including indirect charges. Total Cost of your ' . Config::get('constants.cases_type.' . $case->case_type) . ' is Rs. ' . $pagesCharges . '/-. and Service Charges are Rs. ' . $serviceCharges . '/-. Total amount to pay for your attested copy is Rs. ' . $serviceCharges . ' + ' . $pagesCharges . ' = Rs. ' . ($serviceCharges + $pagesCharges) . '/-. Kindly Pay Rs. ' . ($serviceCharges + $pagesCharges) . '/- in the Easypaisa account of the Attorney Official.';

                    $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about case details payments',
                        'attorney_case_update', $message, null, ($serviceCharges + $pagesCharges));
                }

            }
            if ($caseCustomer) {
                PushNotification::sendPushNotification('Case Update', $message, $caseCustomer->customer_id);
            }
            return $this->success(200, null, 'message sent to the customer successfully');
        }
    }

    public function attorneyUpdateAttestedCaseDetails(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();
            $castCostDetails = CaseCost::where('case_type', $case->case_type)->first();

            if ($case->case_type == 1 || $case->case_type == 2 || $case->case_type == 3 || $case->case_type == 4) {
                if ($case->case_type == 4) {
                    if ($case->case_status == 'Pending') {
                        $serviceCharges = $castCostDetails->pending_service_charges;
                    } else {
                        $serviceCharges = $castCostDetails->decided_service_charges;
                    }
                } else {
                    if ($case->case_status == 'Pending') {
                        $serviceCharges = $castCostDetails->pending_service_charges;
                    } else {
                        $serviceCharges = $castCostDetails->decided_service_charges;
                    }
                }
                $perPageCharges = $castCostDetails->per_page_cost;
                $pagesCharges = $perPageCharges * $request->pages;

//                $message = 'Your per page cost of ' . Config::get('constants.cases_type.' . $case->case_type) . ' including Indirect Charges is Rs. ' . $perPageCharges . '/- Total Number of your Attested Copy are ' . $request->pages . ' pages Total Cost of Copies is Rs. ' . $pagesCharges . '/- Service Charges Rs. ' . $serviceCharges . '/- Total Charges are ' . $serviceCharges . ' + ' . $pagesCharges . ' = ' . $chargesWithOutDelivery . '/-';
                $message = 'Total number of pages of your attested copy are ' . $request['pages'] . ' Pages. Attested Copy Per Page Charges is Rs. ' . $perPageCharges . '/- including indirect charges. Total Cost of your ' . Config::get('constants.cases_type.' . $case->case_type) . ' is Rs. ' . $pagesCharges . '/-. and Service Charges are Rs. ' . $serviceCharges . '/-. Total amount to pay for your attested copy is Rs. '
                    . $serviceCharges . ' + ' . $pagesCharges . ' = Rs. ' . ($serviceCharges + $pagesCharges) . '/-. Kindly Pay Rs. ' . ($serviceCharges + $pagesCharges) . '/- in the Easypaisa account of the Attorney Official.';
                PushNotification::sendPushNotification('Case Update', $message, $case->customer_id);
                $amount = $serviceCharges + $pagesCharges;
                $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about case details payments',
                    'attorney_case_pages_update', $message, null, $amount);
            }

            return $this->success(200, null, 'message sent to the customer successfully');
        }
    }

    public function attorneyUpdateNewsPaperCharges(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $caseCustomer = CustomerCase::where('id', $request->case_id)->first();
            $message = 'Kindly transfer Rs. ' . $request->charges . '/- for newspaper advertisement in official bank accounts of Attorney Official as provided in the Payment Method options in main screen of the app.';
            $this->updateCaseDetails($request->case_id, Auth::id(), 'attorney', 'Update the customer about case details payments',
                'attorney_case_news_paper_charges_update', $message, null, $request->charges);
            PushNotification::sendPushNotification('Case Update', $message, $caseCustomer->customer_id);
            return $this->success(200, null, 'message sent to the customer successfully');
        }
    }

    public function customerWantDelivery(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
            'delivery_type' => 'required'
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();
            $district = District::where(['type' => 1, 'name' => $case->court_name])->first();
            $customer = Customer::where('id', $case->customer_id)->first();
            $message = '';
            if ($request->delivery_type == 1) {
                $message = 'Attorney will now deliver your documents copy to your shipping address ' . $customer->shipping_address . ' through cash on delivery courier service and upload the picture of courier receipt for tracking your parcel';
                $this->updateCaseDetails($request->case_id, Auth::id(),
                    'customer', 'want to deliver on address', 'customer_needs_delivery', $message);
            } else if ($request->delivery_type == 2) {
                $message = 'You will collect the documents from this address ' . $district->address . ' Contact No.: ' . $district->office_mobile;
                $this->updateCaseDetails($request->case_id, Auth::id(),
                    'customer', 'want to deliver on address', 'customer_will_pick_from_office', $message);
            }
            PushNotification::sendPushNotification('Case Update', $message, $case->attorney_id);
            return $this->success(200, null, 'message sent to the attorney successfully');
        }
    }

    public function updateDeliveryByAttorney(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'case_id' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->error(400, $validator->errors(), 'The given data was invalid');
        } else {
            $case = CustomerCase::where('id', $request->case_id)->first();

            $this->updateCaseStatus($request->case_id, 2);
            $caseDetails = $this->updateCaseDetails($request->case_id, Auth::id(),
                'attorney', 'couriered successfully', 'upload_delivery_receipt', 'Your file has been couriered successfully');
            $message = 'Your file has been couriered successfully';
            PushNotification::sendPushNotification('Case Update', $message, $case->customer_id);
            if ($request->has('delivery_receipt_image')) {
                $imageUrl = $this->addCaseImageData($request->delivery_receipt_image);
                Image::create([
                    'imageable_type' => get_class(new CaseDetail()),
                    'imageable_id' => $caseDetails->id,
                    'url' => $imageUrl,
                    'type' => 'delivery_receipt_image',
                ]);
            }

            return $this->success(200, null, 'message sent to the user successfully');
        }
    }

    public function casePaymentEasyPaisa(Request $request)
    {

//        $casePayment = new CasePayment();
//        $casePayment->case_id = $request->order_id;
//        $casePayment->transaction_id = 'transaction_id';
//        $casePayment->transaction_date_time = 'transactionDateTime';
//        $casePayment->amount = $request->amount;
//        $casePayment->response_code = 'responseCode';
//        $casePayment->response_desc = 'SUCCESS';
//        $casePayment->save();
//        return $this->success(200, $casePayment, '$data->responseDesc');
//        03445813239;
        $types = ['attorney_case_news_paper_charges_update', 'request', 'attorney_case_pages_update', 'attorney_case_update'];
        $caseDetails = CaseDetail::where('case_id', $request->order_id)->whereIn('type', $types)->orderBy('id', 'desc')->first();

        $data = [
            "orderId" => $caseDetails->id,
            // "storeId" => "22196",
            "storeId" => "417746",
            "transactionAmount" => $request->amount,
            "transactionType" => "MA",
            "mobileAccountNo" => $request->mobile,
            "emailAddress" => $request->email
        ];
//        dd($data);
        $response = Http::
        withHeaders(['Credentials' => env('EASE_PAISA_CREDENTIALS_URL')])
            ->post(env('EASE_PAISA_PAYMENT_LIVE_URL'), $data);
        // dd($response->body());

        if ($response->status() == 200) {
            $data = $response->body();
            $data = \GuzzleHttp\json_decode($data);
            $casePayment = new CasePayment();
            $casePayment->case_id = $data->orderId;
            $casePayment->transaction_id = $data->transactionId ?? '0';
            $casePayment->transaction_date_time = $data->transactionDateTime;
            $casePayment->amount = $request->amount;
            $casePayment->response_code = $data->responseCode;
            $casePayment->response_desc = $data->responseDesc;
            $casePayment->save();

            return $this->success(200, $casePayment, $data->responseDesc);
        } else {
            $data = $response->body();
            $data = \GuzzleHttp\json_decode($data);
            $casePayment = new CasePayment();
            $casePayment->case_id = $data->orderId;
            $casePayment->transaction_id = '0';
            $casePayment->amount = $request->amount;
            $casePayment->transaction_date_time = $data->transactionDateTime;
            $casePayment->response_code = $data->responseCode;
            $casePayment->response_desc = $data->responseDesc;
            $casePayment->save();
            return $this->success($response->status(), $casePayment, $data['responseDesc']);
        }
    }
}
